// Program przesyla dane z przetwornika AC do pamieci RAM kanalem DMA.
//
// Do wejscia przetwornika AC podlaczony jest wewnetrzny (w strukturze mikrokontrolera) czujnik temperatury.
//
// Zasada dzialania programu:
//
// 1. Cykliczne przepelnienie timera TIM3 generuje sygnal TRGO (Trigger Output) inicjujacy konwersje AC.
// 2. Po zakonczonej konwersji przetwornik AC zglasza zadanie (request) transferu danych kanalem DMA.
// 3. Uklad DMA przesyla dane z Data Register (ADC1->DR) przetwornika AC do pierwszego (i kolejnych)
//    elementu(w) tablicy "buff" (w pamieci RAM).
//
// Zadania.
//
// 1. Skonfigurowac uklad DMA. Ustawic adresy odczytu i zapisu danych, uruchomic program w trybie
//    pojedynczych transferow sampli z przetwornika ADC1 do kolejnych elementow buff[i] (i = 0 ... 7),
//		(jedna konwersja -> jedna probka -> jeden transfer do pamieci pod odpowiednio inkrementowany adres).
//
//		-> Wyswietlic 8 kolejnych odczytow temperatury. Zaobserowac zachowanie ukladu.
//
// 2. Zmienic rozdzielczosc przetwornika AC z 10 na 12 bitow; obliczyc LSB.
//
//		-> Wyswietlic 8 kolejnych odczytow temperatury. Zaobserowac zachowanie ukladu.
//
// 3. Wlaczyc i ustawic kolejke FIFO w ukladzie DMA i wykorzystac transfery seriami
//    (zgromadzenie np. 8 sampli 12(16) bitowych w buforze FIFO -> jeden transfer calosci do pamieci).
//
//		-> Wyswietlic wszystkie temperatury - analogicznie jak w poprzednim punkcie. Co sie zmienilo?
//
// 4. Obliczyc srednia arytmetyczna z 8 ostatnich odczytow i ja wypisac -> sprawdzic wartosc.
//
// 5. Opcjonalnie: zmierzyc rzeczwista wartosc napiecia odniesienia Vref przetwornika ADC1 (domyslnie Vref = 3.0 V).
//		Skorygowac skalowanie temperatury, nastepnie sprawdzic wplyw zmiany Vref na poprawnosc pomiarow.

#include <stdio.h>
#include "stm32f4xx.h"
#include "GLCD.h"

// --- 5 ---
// Nalezy sprawdzic i zmierzyc Vref kazdego egzemplarza zestawu Discovery!

#define	Vref 3.0f

// --- 2b ---
// Skorygowac LSB, w zaleznosci od rozdzielczosci przetwornika.

#define LSB Vref/1024.0f

float t;
uint32_t i;
uint16_t buff[8];
char str[20];

int main(void){

	GLCD_Initialize();
	GLCD_Clear(Black);                        
	GLCD_SetBackColor(Black);
  GLCD_SetTextColor(White);
	
  RCC->AHB1ENR |= (1<<22); // Wlacz "zegar" DMA2.
	RCC->APB1ENR |= (1<<1);  // Wlacz "zegar" TIM3.
	RCC->APB2ENR |= (1<<8);  // Wlacz "zegar" ADC1.

// --- 1 ---

// Ustawienia DMA (rozdzial 10.5 w Reference Manual, ew. Application Note AN4031).

// Konfiguracja parametrow transferu danych:
	
// 32-bitowy adres urzadzenia peryferyjnego (Peripheral Address Reg.):

//	DMA2_Stream0->PAR =
	
// 32-bitowy adres miejsca w pamieci RAM do ktorego beda zapisywane dane (Memory Address Reg.):	

//	DMA2_Stream0->M0AR =
	
// Liczba przesylanych danych (probek) w jednej sekwencji:

//	DMA2_Stream0->NDTR =

// Opis rejestru SxCR - rozdzial 10.5.5 w RM. W szczegolnosci nalezy zwrocic uwage na: 
// kierunek przeplywu danych,
// typy/rozmiar danych, inkrementacje wskaznikow,
// wlaczyc tryb pierscieniowy (circular)!

//	DMA2_Stream0->CR =
	
// --- 3 ---

// Ustawic pelna pojemnosc i wlaczyc bufor FIFO, wylaczyc Direct Mode.
// 0pis rejestru SxFCR - rozdzial 10.5.10 w RM.

//	DMA2_Stream0->FCR = 
	
	DMA2_Stream0->CR |= 1;											// Wlacz DMA.
	
// --- 2a ---
// Zmienic rozdzielczosc przetwornika AC z 10 na 12 bitow.
// Opis rejestru CR1 (Control Register 1) - rozdzial 13.13.2 w Reference Manual.

	ADC1->CR1= 1 << 24;													// Rozdzielczosc przetwornika AC.

// Pozostala konfiguracja przetwornika AC - (zrobiona).
	ADC1->CR2 = (1<<28)|(8<<24)|(1<<9)|(1<<8);	// External TIM3 TRGO Trigger/wlacz DMA.
	ADC1->SQR1 = 1<<20;													// Pojedyncza konwersja AC.
	ADC1->SQR3 = 18; 														// Kanal 18.	
	ADC->CCR = (1<<23)|(3<<16);									// Wlacz Temp_Sensor; ADC prescaler = /8.
	ADC1->SMPR1 |= 7<<24;												// Sampling time - maks. - 480 taktow.
	ADC1->CR2 |= 1;															// Wlacz ADC1.

// Konfiguracja timera TIM3. Czestotliwosc probkowania 1 Hz:

	TIM3->PSC = 15999;
	TIM3->ARR = 999;
	TIM3->CR2 |= 2<<4;													// W chwili przepelnienia -> TRGO.
	TIM3->CR1 |= 1;															// Wlacz timer.

	for(;;){
				
// Odczyt wew. czujnika temperatury i obliczenie jej wartosci.
// Surowe sample z ADC sa zapisane przez uklad DMA w buff[i].
// Rozdzial 13.10 Reference Manual, parametry w Datasheet 6.3.22.

// --- 4 ---
// Dopisac: usrednianie ostatnich 8 odczytow temperatury.
 				
			t=(( (LSB * (float)buff[i]) - 0.76f)  * 400.0f ) + 25.0f;
			sprintf(str,"t[%u] = %2.2f %cC     ",i,t,0x82);
			GLCD_DisplayString(i,0,str);
			
			i++; i &= 0x07;
		
	}
}
